/*
 Copyright 2004-2008 Paul R. Holser, Jr.  All rights reserved.
 Licensed under the Academic Free License version 3.0
 */

package joptsimple;

import java.util.List;

/**
 * <p>Wraps an option specification, and allows callers to specify whether the option
 * accepts arguments (required or optional).</p>
 *
 * <p>Instances are returned from {@link OptionParser#accepts(String)} to allow the
 * formation of parser directives as sentences in a domain-specific language.  For
 * example:</p>
 *
 * <pre><code>
 *   OptionParser parser = new OptionParser();
 *   parser.accepts( "c" ).<strong>withRequiredArg()</strong>.ofType( Integer.class );
 * </code></pre>
 *
 * <p>If no methods are invoked on an instance of this class, then that instance's option
 * will accept no argument.</p>
 *
 * @since 2.0
 * @author <a href="mailto:pholser@alumni.rice.edu">Paul Holser</a>
 * @version $Id: OptionSpecBuilder.java,v 1.2 2008/04/08 03:20:17 pholser Exp $
 */
public class OptionSpecBuilder {
    private final OptionParser parser;
    private final List options;
    private final String description;

    OptionSpecBuilder( OptionParser parser, List options, String description ) {
        this.parser = parser;
        this.options = options;
        this.description = description;

        parser.recognize( new NoArgumentOptionSpec( options, description ) );
    }

    /**
     * <p>Informs an option parser that this builder's option requires an argument.</p>
     *
     * @return a specification for the option
     */
    public ArgumentAcceptingOptionSpec withRequiredArg() {
        ArgumentAcceptingOptionSpec newSpec =
            new RequiredArgumentOptionSpec( options, description );
        parser.recognize( newSpec );

        return newSpec;
    }

    /**
     * <p>Informs an option parser that this builder's option accepts an optional
     * argument.</p>
     *
     * @return a specification for the option
     */
    public ArgumentAcceptingOptionSpec withOptionalArg() {
        ArgumentAcceptingOptionSpec newSpec =
            new OptionalArgumentOptionSpec( options, description );
        parser.recognize( newSpec );

        return newSpec;
    }
}
