/*
Copyright 2012-2014 Samuel Gesche

This file is part of the Greek Reuse Toolkit.

The Greek Reuse Toolkit is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

The Greek Reuse Toolkit is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with the Greek Reuse Toolkit.  If not, see <http://www.gnu.org/licenses/>.
*/

package fr.cnrs.liris.drim.grt.proc.parsers;

import fr.cnrs.liris.drim.grt.ihm.GestionnaireErreurs;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.text.BadLocationException;
import javax.swing.text.rtf.RTFEditorKit;
import fr.cnrs.liris.drim.grt.modele.Coordonnee;
import fr.cnrs.liris.drim.grt.modele.OrdreStrict;
import fr.cnrs.liris.drim.grt.modele.Passage;
import fr.cnrs.liris.drim.grt.modele.Reference;
import fr.cnrs.liris.drim.grt.modele.Terme;
import fr.cnrs.liris.drim.grt.modele.convertisseurs.ConvertisseurBetaCodeVersUnicode;
import fr.cnrs.liris.drim.grt.modele.convertisseurs.ConvertisseurBibleWorksVersUnicode;
import fr.cnrs.liris.drim.grt.modele.convertisseurs.ConvertisseurTexteUnicodeVersUnicode;
import fr.cnrs.liris.drim.grt.modele.lemmatiseurs.BibleWorks;
import fr.cnrs.liris.drim.grt.modele.lemmatiseurs.Perseus;
import fr.cnrs.liris.drim.grt.modele.lemmatiseurs.SourcesChretiennes;
import fr.cnrs.liris.drim.grt.modele.listes.NomsPropres;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import fr.cnrs.liris.drim.grt.proc.ReferencesDeCitation;
import fr.cnrs.liris.drim.grt.proc.exceptions.DOMCreationFailureException;

/**
 *
 * @author sgesche
 */
public class ImportFichier {
    
    // plus utilisé ; si réutilisé, à remettre à jour
    
    /*public static Document importBibleTSV(File fichierTSV) throws IOException {
        LecteurGrec reader = new LecteurGrec(new InputStreamReader(new FileInputStream(fichierTSV), "UTF-8"));
        Document resultat = new Document(fichierTSV.getName());
        Passage livreCourant = null; // normalement, ne doit pas servir avant affectation
        while(reader.ready()) {
            String ligne = reader.readLine();
            String[] champs = ligne.split("\t");
            String t = champs[6];
            if(t.contains("t")) {
                String s = champs[7].replaceAll("\\\"", "");
                livreCourant = new Passage(s, "");
                resultat.addSousPassage(livreCourant);
                // traiter comme un nouveau livre de titre [7]
            } else {
                try {
                    t = t.replaceAll("\\\"", "");
                    String s = champs[7];
                    if(!Textes.estVideDeTermes(s)) {
                        livreCourant.addSousPassage(new Passage(t, s));
                    }
                } catch(NullPointerException ex) {
                    GestionnaireErreurs.getInstance().affiche("Erreur d'application", 
                            "<html><p>Nous avons un verset que nous n'arrivons pas à rattacher à un livre, au début du fichier.</p>"
                            + "<p>Son contenu : "+ligne+"</p>"
                            + "<p>Pour information, l'erreur est</p>"
                            + "<p>"+ex.getClass()+" : "+ex.getMessage()+"</p></html>");
                }
            }
        }
        return resultat;
    }*/
    
    public static Passage importPatristiqueRTF(File fichierRTF) throws IOException, BadLocationException {
        BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(fichierRTF), "UTF-8"));
        Passage resultat = Passage.creeDocument(fichierRTF.getName());
        RTFEditorKit editor = new RTFEditorKit();
        javax.swing.text.Document doc = new javax.swing.text.DefaultStyledDocument();
        editor.read(reader, doc, 0);
        String[] contenu = ConvertisseurTexteUnicodeVersUnicode.conversion(doc.getText(0, doc.getLength()).replaceAll("[<>]", "").replaceAll(" {0,}\\.{2,} {0,2}", " "), false).split("(\\r?\\n)|\\s{2,}");
        
        
        boolean corpsDeTexte = true;
        boolean titreInitial = true;
        
        int chapitreCourant = 1;
        int paragrapheCourant = 1;
        int ligneCourante = 1;
        
        boolean ligneCoupee = false;
        Passage contenuLigneCoupee = null;
        String motCoupe = "";
        int nbMotsLigneCoupee = 0;
        
        for (String contenu1 : contenu) {
            String ligne = contenu1.trim();
            //System.out.println(ligne);
            if(Textes.estVideDeTermes(ligne)) {
                // rien : une ligne vide ne doit pas être comptabilisée
                //System.out.println("  ligne vide");
            } else if(isTitreSection(ligne)) {
                //System.out.println("  titre");
                titreInitial = false;
                // enregistrer le titre dans la section courante (chapitre ? paragraphe ?)
                corpsDeTexte = false;
            } else if(isNumero(ligne)) {
                //System.out.print("  numéro");
                titreInitial = false;
                String[] nums = ligne.split("\\.");
                //System.out.println(" ->" + Arrays.toString(nums));
                switch(nums.length) {
                    case 4:
                        chapitreCourant = Integer.parseInt(nums[1].trim());
                        paragrapheCourant = Integer.parseInt(nums[2].trim());
                        ligneCourante = Integer.parseInt(nums[3].trim());
                        break;
                    case 3:
                        chapitreCourant = Integer.parseInt(nums[0].trim());
                        paragrapheCourant = Integer.parseInt(nums[1].trim());
                        ligneCourante = Integer.parseInt(nums[2].trim());
                        break;
                    case 2:
                        paragrapheCourant = Integer.parseInt(nums[0].trim());
                        ligneCourante = Integer.parseInt(nums[1].trim());
                        break;
                    case 1:
                        ligneCourante = Integer.parseInt(nums[0].trim());
                        break;
                    default:
                        // rien... à compléter suivant d'autres cas
                }
                corpsDeTexte = true;
            } else {
                //System.out.print("  texte");
                ligne = ligne.replaceAll("\\. ", " . ");
                ligne = ligne.replaceAll("; ", " . ");
                ligne = ligne.replaceAll("\\s+?", " ");
                if(titreInitial) {
                    //System.out.println(" -> titre du document");
                    resultat.setTitre(resultat.getTitre() + " - " + ligne.trim());
                } else if(corpsDeTexte) {
                    //System.out.println(" -> ligne");
                    if(ligneCoupee) {
                        String mot;
                        if(ligne.contains(" ")) {
                            mot = ligne.substring(0, ligne.indexOf(" "));
                        } else {
                            mot = ligne;
                        }
                        motCoupe += mot;
                        Passage pMot = Passage.creeMot(new Passage[]{contenuLigneCoupee}, 
                                Terme.cree(motCoupe));
                        contenuLigneCoupee.addSousPassage(new Coordonnee(
                                OrdreStrict.getInstance(OrdreStrict.MOTS), 
                                ""+(nbMotsLigneCoupee+1)), pMot);
                        ligne = ligne.substring(ligne.indexOf(" ")+1, ligne.length());
                        ligneCoupee = false;
                    }
                    String ligne2 = "";
                    if(ligne.trim().endsWith("-")) {
                        String[] listeMots = ligne.split(" ");
                        nbMotsLigneCoupee = listeMots.length-1;
                        for(int j=0; j<nbMotsLigneCoupee; j++) {
                            ligne2 += listeMots[j] + " ";
                        }
                        ligne2 = ligne2.trim();
                        motCoupe = listeMots[listeMots.length-1].replaceAll("\\-", "");
                        ligneCoupee = true;
                    } else {
                        ligne2 = ligne;
                    }
                    Passage nouvelleLigne = new Passage("", new Passage[]{}, ligne2);
                    if(ligneCoupee) {
                        contenuLigneCoupee = nouvelleLigne;
                    }
                    Reference ch = new Reference();
                    ch.setBorne(Reference.COTE_DEBUT, new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant));
                    Reference par = new Reference();
                    par.setBorne(Reference.COTE_DEBUT, new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.PARAGRAPHES), paragrapheCourant));
                    ch.addSousReference(Reference.RELATION_PRECISION, par);
                    resultat.addPassagesUpTo(ch, new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.LIGNES), ligneCourante), nouvelleLigne);
                    ligneCourante ++;
                } else {
                    //System.out.println(" -> titre de passage");
                    // enregistrer le titre dans le passage courant (chapitre ? paragraphe ?)
                }
            }
        }
        /*for(String s: contenu) {
            System.err.println(Textes.normalise(s));
        }*/
        return resultat;
    }
    
    public static Passage importBibleTexteOuMorphoTXT(File fichierTXT) throws IOException {
        BufferedReader reader = new BufferedReader(new FileReader(fichierTXT)/*new InputStreamReader(new FileInputStream(fichierTXT), "ANSI")*/);
        Passage resultat = Passage.creeDocument(fichierTXT.getName());
        Passage livreCourant = new Passage("", new Passage[]{resultat}, "");
        Passage chapitreCourant = new Passage("", new Passage[]{livreCourant}, "");
        while(reader.ready()) {
            String ligne = reader.readLine();
            //System.out.println("Lecture : " + ligne);
            String ligneConvertie = ConvertisseurBibleWorksVersUnicode.conversion(ligne.replaceAll("@\\S+ ", "").trim());
            String ligneConvertieTest = ligne.replaceAll("@\\S+ ", "").trim();
            String[] mots = ligneConvertie.split("\\s");
            String[] motsTest = ligneConvertieTest.split("\\s");
            String[] chapitreEtVerset = ligne.split("\\s")[1].split(":");
            if(!ligne.split("\\s")[0].equals(livreCourant.getTitre())) {
                if(!livreCourant.getTitre().isEmpty()) {
                    resultat.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.LIVRES_BIB), livreCourant.getTitre()), 
                            livreCourant);
                }
                if(!chapitreCourant.getTitre().isEmpty()) {
                    livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
                }
                //System.out.println(" Nouveau nom de livre : " + ligne.split("\\s")[0]);
                livreCourant = new Passage(ligne.split("\\s")[0], new Passage[]{resultat}, new HashMap<Coordonnee, Passage>());
                //System.out.println(" Nouveau numéro de chapitre : " + chapitreEtVerset[0]);
                chapitreCourant = new Passage(chapitreEtVerset[0], new Passage[]{livreCourant}, new HashMap<Coordonnee, Passage>());
                //System.out.println(" Mise du nouveau chapitre dans le nouveau livre");
                chapitreCourant.setParents(new Passage[]{livreCourant});
            } else if(!chapitreEtVerset[0].equals(chapitreCourant.getTitre())) {
                if(!chapitreCourant.getTitre().isEmpty()) {
                    livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
                }
                //System.out.println(" Nouveau numéro de chapitre : " + chapitreEtVerset[0]);
                chapitreCourant = new Passage(chapitreEtVerset[0], new Passage[]{livreCourant}, new HashMap<Coordonnee, Passage>());
            }
            ArrayList<String> motsReels = new ArrayList<>();
            for(int i=2; i<mots.length; i++) {
                String mot = Textes.nettoieTerme(mots[i]);
                
                /*if(mot.endsWith("ύ") || mot.endsWith("ή") || mot.endsWith("ί") || mot.endsWith("έ") || mot.endsWith("ά") || mot.endsWith("ώ") || mot.endsWith("ό")) {
                    System.err.println(mot);
                }*/
                //
                Terme.cree(mot).getLemmePrincipal();
                //
                if(!mot.trim().isEmpty()) {
                    motsReels.add(mot);
                }
                if(!Textes.estUnTermeValide(mot)) {
                    System.err.println("Transformation BibleWorks -> Unicode incomplète : " + motsTest[i] + " dans " + ligne);
                }
            }
            Passage verset = new Passage(chapitreEtVerset[1], new Passage[]{chapitreCourant}, new HashMap<Coordonnee, Passage>());
            for(int i=0; i<motsReels.size(); i++) {
                verset.addSousPassage(
                        new Coordonnee(OrdreStrict.getInstance(OrdreStrict.MOTS), 
                        ""+(i+1)), 
                        Passage.creeMot(new Passage[]{verset}, Terme.cree(motsReels.get(i))));
            }
            chapitreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.VERSETS), verset.getTitre()), 
                            verset);
            //System.out.println(" Entrée du verset : " + chapitreEtVerset[1]);
        }
        if(!chapitreCourant.getTitre().isEmpty()) {
            livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
            //System.out.println(" Entrée du chapitre : " + chapitreCourant.getTitre());
        }
        if(!livreCourant.getTitre().isEmpty()) {
            resultat.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.LIVRES_BIB), livreCourant.getTitre()), 
                            livreCourant);
            //System.out.println(" Entrée du livre : " + livreCourant.getTitre());
        }
        return resultat;
    }
    
    public static Passage importPhilonTexteOuMorphoTXT(File fichierTXT) throws IOException {
        BufferedReader reader = new BufferedReader(new FileReader(fichierTXT)/*new InputStreamReader(new FileInputStream(fichierTXT), "ANSI")*/);
        Passage resultat = Passage.creeDocument(fichierTXT.getName());
        Passage livreCourant = new Passage("", new Passage[]{resultat}, "");
        Passage chapitreCourant = new Passage("", new Passage[]{livreCourant}, "");
        while(reader.ready()) {
            String ligne = reader.readLine();
            //System.out.println("Lecture : " + ligne);
            String ligneConvertie = ConvertisseurBibleWorksVersUnicode.conversion(ligne.replaceAll("@\\S+ ", "").trim());
            String ligneConvertieTest = ligne.replaceAll("@\\S+ ", "").trim();
            String[] mots = ligneConvertie.split("\\s");
            String[] motsTest = ligneConvertieTest.split("\\s");
            String[] chapitreEtVerset = ligne.split("\\s")[1].split(":");
            if(!ligne.split("\\s")[0].equals(livreCourant.getTitre())) {
                if(!livreCourant.getTitre().isEmpty()) {
                    resultat.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.LIVRES_PHI), livreCourant.getTitre()), 
                            livreCourant);
                }
                if(!chapitreCourant.getTitre().isEmpty()) {
                    livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
                }
                //System.out.println(" Nouveau nom de livre : " + ligne.split("\\s")[0]);
                livreCourant = new Passage(ligne.split("\\s")[0], new Passage[]{resultat}, new HashMap<Coordonnee, Passage>());
                //System.out.println(" Nouveau numéro de chapitre : " + chapitreEtVerset[0]);
                chapitreCourant = new Passage(chapitreEtVerset[0], new Passage[]{livreCourant}, new HashMap<Coordonnee, Passage>());
                //System.out.println(" Mise du nouveau chapitre dans le nouveau livre");
                chapitreCourant.setParents(new Passage[]{livreCourant});
            } else if(!chapitreEtVerset[0].equals(chapitreCourant.getTitre())) {
                if(!chapitreCourant.getTitre().isEmpty()) {
                    livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
                }
                //System.out.println(" Nouveau numéro de chapitre : " + chapitreEtVerset[0]);
                chapitreCourant = new Passage(chapitreEtVerset[0], new Passage[]{livreCourant}, new HashMap<Coordonnee, Passage>());
            }
            ArrayList<String> motsReels = new ArrayList<>();
            for(int i=2; i<mots.length; i++) {
                String mot = Textes.nettoieTerme(mots[i]);
                //
                Terme.cree(mot).getLemmePrincipal();
                //
                if(!mot.trim().isEmpty()) {
                    motsReels.add(mot);
                }
                if(!Textes.estUnTermeValide(mot)) {
                    System.err.println("Transformation BibleWorks -> Unicode incomplète : " + motsTest[i] + " dans " + ligne);
                }
            }
            Passage verset = new Passage(chapitreEtVerset[1], new Passage[]{chapitreCourant}, new HashMap<Coordonnee, Passage>());
            for(int i=0; i<motsReels.size(); i++) {
                verset.addSousPassage(
                        new Coordonnee(OrdreStrict.getInstance(OrdreStrict.MOTS), 
                        ""+(i+1)), 
                        Passage.creeMot(new Passage[]{verset}, Terme.cree(motsReels.get(i))));
            }
            chapitreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.VERSETS), verset.getTitre()), 
                            verset);
            //System.out.println(" Entrée du verset : " + chapitreEtVerset[1] + " (" + verset.getNbSousPassages()+ " mots)");
        }
        if(!chapitreCourant.getTitre().isEmpty()) {
            livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
            //System.out.println(" Entrée du chapitre : " + chapitreCourant.getTitre());
        }
        if(!livreCourant.getTitre().isEmpty()) {
            resultat.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.LIVRES_PHI), livreCourant.getTitre()), 
                            livreCourant);
            //System.out.println(" Entrée du livre : " + livreCourant.getTitre());
        }
        return resultat;
    }
    
    public static Passage importTexteOuMorphoTXT(File fichierTXT) throws IOException {
        BufferedReader reader = new BufferedReader(new FileReader(fichierTXT)/*new InputStreamReader(new FileInputStream(fichierTXT), "ANSI")*/);
        Passage resultat = Passage.creeDocument(fichierTXT.getName());
        Passage livreCourant = new Passage("", new Passage[]{resultat}, "");
        Passage chapitreCourant = new Passage("", new Passage[]{livreCourant}, "");
        int noLivreCourant = 0;
        while(reader.ready()) {
            String ligne = reader.readLine();
            //System.out.println("Lecture : " + ligne);
            String ligneConvertie = ConvertisseurBibleWorksVersUnicode.conversion(ligne.replaceAll("@\\S+ ", "").trim());
            String ligneConvertieTest = ligne.replaceAll("@\\S+ ", "").trim();
            String[] mots = ligneConvertie.split("\\s");
            String[] motsTest = ligneConvertieTest.split("\\s");
            String[] chapitreEtVerset = ligne.split("\\s")[1].split(":");
            if(!chapitreEtVerset[0].equals(chapitreCourant.getTitre())) {
                if(!chapitreCourant.getTitre().isEmpty()) {
                    livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
                }
                //System.out.println("Nouveau chapitre : " + chapitreEtVerset[0]);
                chapitreCourant = new Passage(chapitreEtVerset[0], new Passage[]{livreCourant}, new HashMap<Coordonnee, Passage>());
            }
            if(!ligne.split("\\s")[0].equals(livreCourant.getTitre())) {
                if(!livreCourant.getTitre().isEmpty()) {
                    resultat.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.NUMERIQUE_ARABE), noLivreCourant), 
                            livreCourant);
                    noLivreCourant++;
                }
                //System.out.println("Nouveau livre : " + ligne.split("\\s")[0]);
                livreCourant = new Passage(ligne.split("\\s")[0], new Passage[]{resultat}, new HashMap<Coordonnee, Passage>());
                chapitreCourant.setParents(new Passage[]{livreCourant});
            }
            ArrayList<String> motsReels = new ArrayList<>();
            for(int i=2; i<mots.length; i++) {
                String mot = Textes.nettoieTerme(mots[i]);
                if(!mot.trim().isEmpty()) {
                    motsReels.add(mot);
                }
                if(!Textes.estUnTermeValide(mot)) {
                    System.err.println("Transformation BibleWorks -> Unicode incomplète dans le fichier "+fichierTXT.getName()+" : " + motsTest[i] + " -> " + mot + " dans " + ligne);
                }
            }
            Passage verset = new Passage(chapitreEtVerset[1], new Passage[]{chapitreCourant}, new HashMap<Coordonnee, Passage>());
            for(int i=0; i<motsReels.size(); i++) {
                verset.addSousPassage(
                        new Coordonnee(OrdreStrict.getInstance(OrdreStrict.MOTS), 
                        ""+(i+1)), 
                        Passage.creeMot(new Passage[]{verset}, Terme.cree(motsReels.get(i))));
            }
            chapitreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.VERSETS), verset.getTitre()), 
                            verset);
            //System.out.println("Nouveau verset : " + chapitreEtVerset[1]);
        }
        if(!chapitreCourant.getTitre().isEmpty()) {
            livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
            //System.out.println("Nouveau chapitre : " + chapitreCourant.getTitre());
        }
        if(!livreCourant.getTitre().isEmpty()) {
            resultat.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.NUMERIQUE_ARABE), noLivreCourant), 
                            livreCourant);
            noLivreCourant++;
            //System.out.println("Nouveau livre : " + livreCourant.getTitre());
        }
        return resultat;
    }
    
    public static Passage importTestTXT(File fichierTXT) throws IOException {
        BufferedReader reader = new BufferedReader(new FileReader(fichierTXT)/*new InputStreamReader(new FileInputStream(fichierTXT), "ANSI")*/);
        Passage resultat = Passage.creeDocument(fichierTXT.getName());
        Passage livreCourant = new Passage("", new Passage[]{resultat}, "");
        Passage chapitreCourant = new Passage("", new Passage[]{livreCourant}, "");
        while(reader.ready()) {
            String ligne = reader.readLine().replaceAll("﻿", "");
            //System.out.println("Lecture : " + ligne);
            String ligneConvertie = ligne.trim();
            String[] mots = ligneConvertie.split("\\s");
            String[] chapitreEtVerset = ligne.split("\\s")[1].split(":");
            if(!chapitreEtVerset[0].equals(chapitreCourant.getTitre())) {
                if(!chapitreCourant.getTitre().isEmpty()) {
                    livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
                }
                //System.out.println("Nouveau chapitre : " + chapitreEtVerset[0]);
                chapitreCourant = new Passage(chapitreEtVerset[0], new Passage[]{livreCourant}, new HashMap<Coordonnee, Passage>());
            }
            if(!ligne.split("\\s")[0].equals(livreCourant.getTitre())) {
                if(!livreCourant.getTitre().isEmpty()) {
                    resultat.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.LIVRES_BIB), livreCourant.getTitre()), 
                            livreCourant);
                }
                //System.out.println("Nouveau livre : " + ligne.split("\\s")[0]);
                livreCourant = new Passage(ligne.split("\\s")[0], new Passage[]{resultat}, new HashMap<Coordonnee, Passage>());
                chapitreCourant.setParents(new Passage[]{livreCourant});
            }
            ArrayList<String> motsReels = new ArrayList<>();
            for(int i=2; i<mots.length; i++) {
                if(!mots[i].trim().isEmpty()) {
                    motsReels.add(mots[i]);
                }
            }
            Passage verset = new Passage(chapitreEtVerset[1], new Passage[]{chapitreCourant}, new HashMap<Coordonnee, Passage>());
            for(int i=0; i<motsReels.size(); i++) {
                verset.addSousPassage(
                        new Coordonnee(OrdreStrict.getInstance(OrdreStrict.MOTS), 
                        ""+(i+1)), 
                        Passage.creeMot(new Passage[]{verset}, Terme.cree(motsReels.get(i))));
            }
            chapitreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.VERSETS), verset.getTitre()), 
                            verset);
            //System.out.println("Nouveau verset : " + chapitreEtVerset[1]);
        }
        if(!chapitreCourant.getTitre().isEmpty()) {
            livreCourant.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.CHAPITRES), chapitreCourant.getTitre()), 
                            chapitreCourant);
            //System.out.println("Nouveau chapitre : " + chapitreCourant.getTitre());
        }
        if(!livreCourant.getTitre().isEmpty()) {
            resultat.addSousPassage(new Coordonnee(
                            OrdreStrict.getInstance(OrdreStrict.LIVRES_BIB), livreCourant.getTitre()), 
                            livreCourant);
            //System.out.println("Nouveau livre : " + livreCourant.getTitre());
        }
        return resultat;
    }
    
    /**
     * Lit le fichier considéré et en retire une liste de citations.
     * Le fichier est supposé avoir quatre colonnes pour l'instant :
     *  (1) index (pas un identifiant)
     *  (2) statut par rapport au corpus (nouvelle, ancienne, obsolète)
     *  (3) ref citeuse
     *  (4) ref citée
     *  (5) type (littérale, paraphrasée, allusion)
     * La première ligne est le nom des champs
     * @param fichierTSV le fichier
     * @param avecAllusions oui si les allusions doivent être conservées, non sinon
     * @param typeCite le type de passage cité (@see PassageType)
     * @param typeCitant le type de passage citant (@see PassageType)
     * @return 
     */
    public static ReferencesDeCitation[] importCitationsTSV(File fichierTSV, boolean avecAllusions, int typeCite, int typeCitant) throws IOException {
        LecteurGrec reader = new LecteurGrec(new InputStreamReader(new FileInputStream(fichierTSV), "UTF-8"));
        ArrayList<ReferencesDeCitation> resultat = new ArrayList<>();
        
        reader.readLine(); // La première ligne est le nom des champs
        while(reader.ready()) {
            String ligne = reader.readLine();
            String[] champs = ligne.split("\t");
            if(champs.length == 5) {
                String citeuse = champs[2];
                String citee = champs[3];
                String type = champs[4];
                if(!type.equalsIgnoreCase("allusion") || avecAllusions) {
                    resultat.add(new ReferencesDeCitation(
                            References.parseReference(citeuse, typeCitant), 
                            References.parseReference(citee, typeCite)));
                }
            }
        }
        
        ReferencesDeCitation[] res = new ReferencesDeCitation[resultat.size()];
        resultat.toArray(res);
        return res;
    }
    
    public static void importLemmesPerseusEnBetaCode(File fichierTexte) throws IOException {
        BufferedReader r = new BufferedReader(new FileReader(fichierTexte));
        while(r.ready()) {
            String[] entree = r.readLine().split("\t");
            if(entree.length == 2) {
                if(!entree[0].isEmpty() && !entree[1].isEmpty()) {
                    Perseus.getInstance().addLemme(
                            ConvertisseurBetaCodeVersUnicode.conversion(entree[0]), 
                            ConvertisseurBetaCodeVersUnicode.conversion(entree[1]));
                }
            }
        }
        Perseus.getInstance().archive();
    }
    
    /**
     * Augmente la liste des lemmes BibleWorks en utilisant les deux fichiers proposés.
     * Ces deux fichiers doivent contenir le même nombre de mots, dans le même ordre,
     * et donner pour le premier le texte, pour le second la morphologie.
     * Attention : la liste générée n'est pas 100% parfaite : les lemmes par terme 
     * sont fusionnés et ne tiennent donc plus compte du contexte. Pour plus de 
     * fiabilité à ce niveau-là, utiliser la méthode de chargement direct des textes.
     * @param fichierTexte le fichier contenant le texte
     * @param fichierMorpho le fichier contenant la morphologie
     * @throws IOException en cas de problème avec les fichiers
     */
    public static void augmenteListeLemmesBibleWorks(File fichierTexte, File fichierMorpho) throws IOException {
        Terme[] mots = new fr.cnrs.liris.drim.grt.proc.Texte(importTexteOuMorphoTXT(fichierTexte)).getContenu();
        Terme[] lemmes = new fr.cnrs.liris.drim.grt.proc.Texte(importTexteOuMorphoTXT(fichierMorpho)).getContenu();
        if(mots.length != lemmes.length) {
            GestionnaireErreurs.getInstance().affiche("La lemmatisation risque d'être polluée...", 
                    "<html><p>La taille des deux textes diffère ("+mots.length+" mots pour "+lemmes.length+" lemmes).</p>"
                    + "<p>Cela signifie qu'il risque d'y avoir un décalage : vérifiez lemmesBibleWorks.txt après coup.</p></html>");
        }
        for(int i=0; i<mots.length; i++) {
            BibleWorks.getInstance().addLemme(mots[i].getExpression(), lemmes[i].getExpression());
        }
        BibleWorks.getInstance().archive();
    }
    
    /**
     * Augmente la liste des lemmes Source Chrétiennes en utilisant le fichier proposé.
     * Ce fichier est un fichier texte composé de lignes comme suit : 
     * [noLigne] mot [lemme] mot [lemme] ...
     * (s'il y a plusieurs lemmes possibles pour un mot, utiliser mot [lemme1,lemme2...])
     * @param fichierMorpho le fichier contenant la morphologie
     * @throws IOException en cas de problème avec les fichiers
     */
    public static void augmenteListeLemmesTXT(File fichierMorpho) throws IOException {
        LecteurGrec lecteur = new LecteurGrec(new FileReader(fichierMorpho));
        int no = 0;
        while(lecteur.ready()) {
            no++;
            String[] ligne = ConvertisseurTexteUnicodeVersUnicode.conversion(lecteur.readLine().trim(), false).split(" ");
            //System.err.println("Lecture : "+Arrays.toString(ligne));
            // ligne[0] est le numéro de ligne
            if(ligne.length>1) {
                String mot = "";
                ArrayList<String> lemmes = new ArrayList<>();
                for(int i=1; i<ligne.length; i++) {
                    if(ligne[i].startsWith("[")) {
                        String[] listeLemmes = ligne[i].substring(1, ligne[i].length()-1).trim().split(",");
                        for(String lemme: listeLemmes) {
                            if(!lemme.trim().isEmpty()) {
                                lemmes.add(lemme.trim());
                                //System.err.println(" Lemme trouve : "+lemme.trim());
                            }
                        }
                    } else {
                        mot = ligne[i].trim();
                        //System.err.println(" Mot trouve : "+mot);
                    }
                    if(!mot.isEmpty() && !lemmes.isEmpty()) {
                        for(String lemme: lemmes) {
                            //System.err.println("  Ajout : "+mot+"="+lemme);
                            SourcesChretiennes.getInstance().addLemme(mot, lemme);
                        }
                        mot = "";
                        lemmes.clear();
                    }
                }
            }
        }
        SourcesChretiennes.getInstance().archive();
    }

    public static void augmenteListeNomsPropresBibleWorks(File morphologie) throws IOException {
        LecteurGrec reader = new LecteurGrec(new FileReader(morphologie));
        while(reader.ready()) {
            String ligne = reader.readLine();
            String[] mots = ligne.split("\\s");
            for(int i=0; i<mots.length; i++) {
                if(mots[i].trim().startsWith("@n") && mots[i].trim().endsWith("p")) {
                    NomsPropres.getInstance().addNomPropre(Terme.cree(ConvertisseurBibleWorksVersUnicode.conversion(mots[i-1])));
                }
            }
        }
        NomsPropres.getInstance().archive();
    }
    
    private static boolean isNumero(String ligne) {
        Pattern motif = Pattern.compile("[0-9]+?(\\.[0-9]+?)+?");
        Matcher chercheur = motif.matcher(ligne.trim());
        return chercheur.matches();
    }
    
    private static boolean isTitreSection(String ligne) {
        Pattern motif = Pattern.compile("[0-9]+?\\..*?t.*?");
        Matcher chercheur = motif.matcher(ligne.trim());
        Pattern motif2 = Pattern.compile(".*?t.*?[0-9]+?");
        Matcher chercheur2 = motif2.matcher(ligne.trim());
        return chercheur.matches() || chercheur2.matches();
    }
    
    public static void importCorrespondanceCADPTLG(File fichierTSV) throws IOException {
        LecteurGrec reader = new LecteurGrec(new InputStreamReader(new FileInputStream(fichierTSV), "UTF-8"));
        while(reader.ready()) {
            String ligne = reader.readLine();
            String[] champs = ligne.split("\t");
            if(champs.length>12) {
                // colonne 0 : n° CADP
                String cadp = champs[0];
                // colonne 12 : n° TLG
                String tlg = champs[12];
                CorrespondanceCADPTLG.getInstance().setCorrespondance(cadp, tlg);
            }
        }
    }
    
    public static void importCorrespondanceJERLXX(File fichierTSV) throws IOException {
        LecteurGrec reader = new LecteurGrec(new InputStreamReader(new FileInputStream(fichierTSV), "UTF-8"));
        while(reader.ready()) {
            String ligne = reader.readLine();
            String[] champs = ligne.split("\t");
            if(champs.length>9) {
                // colonne 4 : nom LXX
                String lxx = champs[4];
                // colonne 20 : nom JER
                String jer = champs[20];
                CorrespondanceJERLXX.getInstance().setCorrespondance(jer, lxx);
            }
        }
    }
    
    public static String importeRemarques(File origine) throws IOException {
        origine.createNewFile();
        String resultat = "";
        LecteurGrec reader = new LecteurGrec(new InputStreamReader(new FileInputStream(origine), "UTF-8"));
        while(reader.ready()) {
            resultat += reader.readLine() + "\n";
        }
        return resultat;
    }
    
    public static Terme[] importeListeFormes(File origine) throws IOException {
        origine.createNewFile();
        LecteurGrec reader = new LecteurGrec(new InputStreamReader(new FileInputStream(origine), "UTF-8"));
        ArrayList<Terme> liste = new ArrayList<>();
        while(reader.ready()) {
            liste.add(Terme.cree(reader.readLine()));
        }
        Terme[] liste2 = new Terme[liste.size()];
        liste.toArray(liste2);
        return liste2;
    }
    
    /*
        <!DOCTYPE DocBible [
            <!ELEMENT DocBible (Bible+)>
            <!ELEMENT Bible (Livre+)>
                    <!ATTLIST Bible nom CDATA #REQUIRED>
            <!ELEMENT Livre (Chapitre+)>
                    <!ATTLIST Livre nom CDATA #REQUIRED>
            <!ELEMENT Chapitre (Verset+)>
                    <!ATTLIST Chapitre no CDATA #REQUIRED>
            <!ELEMENT Verset (Mot*)>
                    <!ATTLIST Verset no CDATA #REQUIRED>
            <!ELEMENT Mot (#PCDATA)>
                    <!ATTLIST Mot no CDATA #REQUIRED>
        ]>


        <!DOCTYPE DocOeuvres [
            <!ELEMENT DocOeuvres (Oeuvre+)>
            <!ELEMENT Oeuvre (Chapitre+)>
                    <!ATTLIST Oeuvre nom CDATA #REQUIRED>
            <!ELEMENT Chapitre (Paragraphe+)>
                    <!ATTLIST Chapitre no CDATA #REQUIRED>
            <!ELEMENT Paragraphe (Ligne+)>
                    <!ATTLIST Paragraphe no CDATA #REQUIRED>
            <!ELEMENT Ligne (Mot*)>
                    <!ATTLIST Ligne no CDATA #REQUIRED>
            <!ELEMENT Mot (#PCDATA)>
                    <!ATTLIST Mot no CDATA #REQUIRED>
        ]>
    
        <!DOCTYPE DocBible [
            <!ELEMENT DocBible (Bible+)>
            <!ELEMENT Bible (Book+)>
                    <!ATTLIST Bible name CDATA #REQUIRED>
            <!ELEMENT Book (Chapter+)>
                    <!ATTLIST Book name CDATA #REQUIRED>
            <!ELEMENT Chapter (Verse+)>
                    <!ATTLIST Chapter num CDATA #REQUIRED>
            <!ELEMENT Verse (Word*)>
                    <!ATTLIST Verse num CDATA #REQUIRED>
            <!ELEMENT Word (#PCDATA)>
                    <!ATTLIST Word num CDATA #REQUIRED>
        ]>


        <!DOCTYPE DocWorks [
            <!ELEMENT DocWorks (Work+)>
            <!ELEMENT Work (Chapter+)>
                    <!ATTLIST Work name CDATA #REQUIRED>
            <!ELEMENT Chapter (Paragraph+)>
                    <!ATTLIST Chapter num CDATA #REQUIRED>
            <!ELEMENT Paragraph (Line+)>
                    <!ATTLIST Paragraph num CDATA #REQUIRED>
            <!ELEMENT Line (Word*)>
                    <!ATTLIST Line num CDATA #REQUIRED>
            <!ELEMENT Word (#PCDATA)>
                    <!ATTLIST Word num CDATA #REQUIRED>
        ]>
     */
    public static Passage importUnicodeXML(File fichierXML) throws IllegalArgumentException, IOException, DOMCreationFailureException {
        Passage resultat = Passage.creeDocument(fichierXML.getName());
        Document dom = XMLReader.getDOM(fichierXML);
        NodeList root = dom.getElementsByTagName("DocBible");
        if(root.getLength() < 1) {
            root = dom.getElementsByTagName("DocOeuvres");
        }
        if(root.getLength() < 1) {
            root = dom.getElementsByTagName("DocWorks");
        }
        if(root.getLength() < 1) {
            throw new IllegalArgumentException("Le fichier XML fourni ne respecte aucune des DTD du logiciel.");
        }
        remplisPassage(resultat, root);
        return resultat;
    }
    
    private static void remplisPassage(Passage aRemplir, NodeList remplissage) {
        for(int i=0; i<remplissage.getLength(); i++) {
            Node element = remplissage.item(i);
            switch(element.getNodeName()) {
                case "DocBible":
                    remplisPassage(aRemplir, element.getChildNodes());
                    break;
                case "DocOeuvres":
                case "DocWorks":
                    remplisPassage(aRemplir, element.getChildNodes());
                    break;
                case "Bible":
                    remplisPassage(aRemplir, element.getChildNodes());
                    break;
                case "Oeuvre":
                case "Work":
                    remplisPassage(aRemplir, element.getChildNodes());
                    break;
                case "Livre":
                case "Book":
                    String nomLivre;
                    try {
                        nomLivre = element.getAttributes().getNamedItem("nom").getNodeValue();
                    } catch(NullPointerException ex) {
                        nomLivre = element.getAttributes().getNamedItem("name").getNodeValue();
                    }
                    Passage livre = new Passage(nomLivre, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.LIVRES_BIB), nomLivre), livre);
                    remplisPassage(livre, element.getChildNodes());
                    break;
                case "Chapitre":
                case "Chapter":
                    String noCh;
                    try {
                        noCh = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noCh = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    Passage chapitre = new Passage(noCh, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.CHAPITRES), noCh), chapitre);
                    remplisPassage(chapitre, element.getChildNodes());
                    break;
                case "Verset":
                case "Verse":
                    String noV;
                    try {
                        noV = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noV = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    Passage verset = new Passage(noV, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.VERSETS), noV), verset);
                    remplisPassage(verset, element.getChildNodes());
                    break;
                case "Paragraphe":
                case "Paragraph":
                    String noPar;
                    try {
                        noPar = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noPar = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    Passage paragraphe = new Passage(noPar, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.PARAGRAPHES), noPar), paragraphe);
                    remplisPassage(paragraphe, element.getChildNodes());
                    break;
                case "Ligne":
                case "Line":
                    String noLigne;
                    try {
                        noLigne = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noLigne = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    Passage ligne = new Passage(noLigne, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.LIGNES), noLigne), ligne);
                    remplisPassage(ligne, element.getChildNodes());
                    break;
                case "Mot":
                case "Word":
                    String noMot;
                    try {
                        noMot = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noMot = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    String txtMot = element.getChildNodes().item(0).getNodeValue();
                    Passage mot = new Passage(noMot, new Passage[]{aRemplir}, Terme.cree(txtMot));
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.MOTS), noMot), mot);
                    break;
                case "#text":
                    // Rien.
                    break;
                default:
                    System.out.println("Remplissage à venir avec " + element.getNodeName() + " - " + element.getNodeValue() + "(" + element.getNodeType() + ")");
            }
        }
    }
    
    /*
        <!DOCTYPE DocBible [
            <!ELEMENT DocBible (Bible+)>
            <!ELEMENT Bible (Book+)>
                    <!ATTLIST Bible name CDATA #REQUIRED>
            <!ELEMENT Book (Chapter+)>
                    <!ATTLIST Book name CDATA #REQUIRED>
            <!ELEMENT Chapter (Verse+)>
                    <!ATTLIST Chapter num CDATA #REQUIRED>
            <!ELEMENT Verse (Word*)>
                    <!ATTLIST Verse num CDATA #REQUIRED>
            <!ELEMENT Word (#PCDATA)>
                    <!ATTLIST Word num CDATA #REQUIRED>
                    <!ATTLIST Word norm CDATA #IMPLIED>
                    <!ATTLIST Word lem CDATA #IMPLIED>
        ]>


        <!DOCTYPE DocWorks [
            <!ELEMENT DocWorks (Work+)>
            <!ELEMENT Work (Chapter+)>
                    <!ATTLIST Work name CDATA #REQUIRED>
            <!ELEMENT Chapter (Paragraph+)>
                    <!ATTLIST Chapter num CDATA #REQUIRED>
            <!ELEMENT Paragraph (Line+)>
                    <!ATTLIST Paragraph num CDATA #REQUIRED>
            <!ELEMENT Line (Word*)>
                    <!ATTLIST Line num CDATA #REQUIRED>
            <!ELEMENT Word (#PCDATA)>
                    <!ATTLIST Word num CDATA #REQUIRED>
                    <!ATTLIST Word norm CDATA #IMPLIED>
                    <!ATTLIST Word lem CDATA #IMPLIED>
        ]>
     */
    public static PassageType importUnicodeXMLNLP(File fichierXML) throws IllegalArgumentException, IOException, DOMCreationFailureException {
        Passage resultat = Passage.creeDocument(fichierXML.getName());
        int type = PassageType.PASSAGE_BIBLIQUE;
        Document dom = XMLReader.getDOM(fichierXML);
        NodeList root = dom.getElementsByTagName("DocBible");
        if(root.getLength() < 1) {
            root = dom.getElementsByTagName("DocOeuvres");
            type = PassageType.PASSAGE_PATRISTIQUE;
        }
        if(root.getLength() < 1) {
            root = dom.getElementsByTagName("DocWorks");
            type = PassageType.PASSAGE_PATRISTIQUE;
        }
        if(root.getLength() < 1) {
            throw new IllegalArgumentException("The xml file does is not compliant with any of the expected DTDs.");
        }
        remplisPassageNLP(resultat, root);
        return new PassageType(resultat, type);
    }
    
    private static void remplisPassageNLP(Passage aRemplir, NodeList remplissage) {
        for(int i=0; i<remplissage.getLength(); i++) {
            Node element = remplissage.item(i);
            switch(element.getNodeName()) {
                case "DocBible":
                    remplisPassage(aRemplir, element.getChildNodes());
                    break;
                case "DocOeuvres":
                case "DocWorks":
                    remplisPassage(aRemplir, element.getChildNodes());
                    break;
                case "Bible":
                    remplisPassage(aRemplir, element.getChildNodes());
                    break;
                case "Oeuvre":
                case "Work":
                    remplisPassage(aRemplir, element.getChildNodes());
                    break;
                case "Livre":
                case "Book":
                    String nomLivre;
                    try {
                        nomLivre = element.getAttributes().getNamedItem("nom").getNodeValue();
                    } catch(NullPointerException ex) {
                        nomLivre = element.getAttributes().getNamedItem("name").getNodeValue();
                    }
                    Passage livre = new Passage(nomLivre, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.LIVRES_BIB), nomLivre), livre);
                    remplisPassage(livre, element.getChildNodes());
                    break;
                case "Chapitre":
                case "Chapter":
                    String noCh;
                    try {
                        noCh = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noCh = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    Passage chapitre = new Passage(noCh, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.CHAPITRES), noCh), chapitre);
                    remplisPassage(chapitre, element.getChildNodes());
                    break;
                case "Verset":
                case "Verse":
                    String noV;
                    try {
                        noV = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noV = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    Passage verset = new Passage(noV, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.VERSETS), noV), verset);
                    remplisPassage(verset, element.getChildNodes());
                    break;
                case "Paragraphe":
                case "Paragraph":
                    String noPar;
                    try {
                        noPar = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noPar = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    Passage paragraphe = new Passage(noPar, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.PARAGRAPHES), noPar), paragraphe);
                    remplisPassage(paragraphe, element.getChildNodes());
                    break;
                case "Ligne":
                case "Line":
                    String noLigne;
                    try {
                        noLigne = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noLigne = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    Passage ligne = new Passage(noLigne, new Passage[]{aRemplir}, new HashMap<Coordonnee, Passage>());
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.LIGNES), noLigne), ligne);
                    remplisPassage(ligne, element.getChildNodes());
                    break;
                case "Mot":
                case "Word":
                    String noMot;
                    try {
                        noMot = element.getAttributes().getNamedItem("no").getNodeValue();
                    } catch(NullPointerException ex) {
                        noMot = element.getAttributes().getNamedItem("num").getNodeValue();
                    }
                    String txtMot = element.getChildNodes().item(0).getNodeValue();
                    String normMot = element.getAttributes().getNamedItem("norm").getNodeValue();
                    String lemMot = element.getAttributes().getNamedItem("lem").getNodeValue();
                    Passage mot = new Passage(noMot, new Passage[]{aRemplir}, Terme.cree(txtMot, normMot, lemMot));
                    aRemplir.addSousPassage(new Coordonnee(OrdreStrict.getInstance(OrdreStrict.MOTS), noMot), mot);
                    break;
                case "#text":
                    // Rien.
                    break;
                default:
                    System.out.println("Remplissage à venir avec " + element.getNodeName() + " - " + element.getNodeValue() + "(" + element.getNodeType() + ")");
            }
        }
    }
    
}
